<?php

namespace PHPMaker2026\Reimbursement;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authorization\Voter\Vote;
use Symfony\Component\Security\Core\Authorization\Voter\Voter as BaseVoter;

/**
 * API Permission Voter
 */
class ApiPermissionVoter extends BaseVoter
{

    public function __construct(
        protected RequestStack $requestStack,
        protected AdvancedSecurity $security,
    ) {
    }

    protected function supports(string $attribute, mixed $subject): bool
    {
        $request = $this->requestStack->getCurrentRequest();
        return str_starts_with($request->getPathInfo(), '/api/');
    }

    protected function voteOnAttribute(string $attribute, mixed $subject, TokenInterface $token, ?Vote $vote = null): bool
    {
        $request = $this->requestStack->getCurrentRequest();
        $routeParams = $request->attributes->get('_route_params');
        $action = RouteAction($request);
        $table = '';
        $shouldVote = match ($action) {
            Config('API_LOOKUP_ACTION'),
            Config('API_EXPORT_CHART_ACTION'),
            Config('API_2FA_ACTION') => true,
            Config('API_JQUERY_UPLOAD_ACTION') => $request->isMethod('POST'),
            default => false,
        };

        // Check user levels
        if ($shouldVote) {
            if (empty($this->security->UserLevelIDs) || !array_any($this->security->UserLevelIDs, fn (int $id) => $id >= AdvancedSecurity::ANONYMOUS_USER_LEVEL_ID)) {
                $vote?->addReason('Access denied: User levels not set or insufficient');
                return false; // Not authorized
            }
        }

        // Actions for table
        $apiTableActions = [
            Config('API_EXPORT_ACTION'),
            Config('API_LIST_ACTION'),
            Config('API_VIEW_ACTION'),
            Config('API_ADD_ACTION'),
            Config('API_EDIT_ACTION'),
            Config('API_DELETE_ACTION'),
            Config('API_FILE_ACTION')
        ];
        if (in_array($action, $apiTableActions)) {
            $table = $routeParams['table'] ?? Param(Config('API_OBJECT_NAME')); // Get from route or Get/Post
        }

        // Check permission
        if (
            $shouldVote // User levels checked
            || $action == Config('API_JQUERY_UPLOAD_ACTION') && $request->isMethod('GET') // Get image during upload (GET)
            || $action == Config('API_PERMISSIONS_ACTION') && $request->isMethod('GET') // Permissions (GET)
            || $action == Config('API_PERMISSIONS_ACTION') && $request->isMethod('POST') && $this->security->isAdmin() // Permissions (POST)
            || $action == Config('API_UPLOAD_ACTION') && $this->security->isLoggedIn() // Upload
            || $action == Config('API_REGISTER_ACTION') // Register
            || $action == Config('API_METADATA_ACTION') // Metadata
            || $action == Config('API_CHAT_ACTION')  // Chat
        ) {
            return true;
        } elseif (in_array($action, $apiTableActions) && $table != '') { // Table actions
            $this->security->loadTablePermissions($table);
            return $action == Config('API_LIST_ACTION') && $this->security->canList()
                || $action == Config('API_EXPORT_ACTION') && $this->security->canExport()
                || $action == Config('API_VIEW_ACTION') && $this->security->canView()
                || $action == Config('API_ADD_ACTION') && $this->security->canAdd()
                || $action == Config('API_EDIT_ACTION') && $this->security->canEdit()
                || $action == Config('API_DELETE_ACTION') && $this->security->canDelete()
                || $action == Config('API_FILE_ACTION') && ($this->security->canList() || $this->security->canView());
        } elseif ($action == Config('API_EXPORT_ACTION') && IsEmpty($table)) { // Get exported file
            return true; // Check table permission in ExportHandler.php
        } elseif ($action == Config('API_LOOKUP_ACTION')) { // Lookup
            $security = $this->security;
            $canLookup = function ($params) use ($security) {
                $object = $params[Config('API_LOOKUP_PAGE')]; // Get lookup page
                $fieldName = $params[Config('API_FIELD_NAME')]; // Get field name
                $lookupField = Container($object)?->Fields[$fieldName] ?? null;
                if ($lookupField) {
                    $lookup = $lookupField->Lookup;
                    if ($lookup) {
                        $tbl = $lookup->getTable();
                        if ($tbl) {
                            $security->loadTablePermissions($tbl->TableVar);
                            if ($security->canLookup()) {
                                return true;
                            } else {
                                $vote?->addReason('No lookup permission for table: ' . $tbl->Name);
                                return false;
                            }
                        }
                    }
                }
            };
            if ($request->getContentTypeFormat() == 'json') { // Multiple lookup requests in JSON
                $parsedBody = $request->request->all();
                if (is_array($parsedBody)) {
                    return array_find($parsedBody, $canLookup) !== null;
                }
            } else { // Single lookup request
                return $canLookup($request->request->all());
            }
        } elseif ($action == Config('API_PUSH_NOTIFICATION_ACTION')) { // Push notification
            $actn = $routeParams['action'];
            if (in_array($actn, [Config('API_PUSH_NOTIFICATION_SUBSCRIBE'), Config('API_PUSH_NOTIFICATION_DELETE')])) {
                if (Config('PUSH_ANONYMOUS') || $this->security->isLoggedIn()) {
                    return true;
                } else {
                    $vote?->addReason('No permission to subscribe/delete push notification');
                    return false;
                }
            } elseif ($actn == Config('API_PUSH_NOTIFICATION_SEND')) {
                $this->security->loadTablePermissions(Config('SUBSCRIPTION_TABLE_NAME'));
                if ($this->security->canPush()) {
                    return true;
                } else {
                    $vote?->addReason('No permission to send push notification');
                    return false;
                }
            }
        } elseif ($action == Config('API_2FA_ACTION')) { // Two factor authentication
            $actn = $routeParams['action'];
            if (in_array($actn, [Config('API_2FA_CONFIG'), Config('API_2FA_SHOW'), Config('API_2FA_VERIFY'), Config('API_2FA_SEND_OTP')])) {
                if ($this->security->isLoggingIn2FA() || $this->security->isLoggedIn()) {
                    return true;
                } else {
                    $vote?->addReason(sprintf('Failed to perform 2FA action (%s). User is not logging in with 2FA or is not logged in.', $actn));
                    return false;
                }
            } elseif (in_array($actn, [Config('API_2FA_BACKUP_CODES'), Config('API_2FA_NEW_BACKUP_CODES'), Config('API_2FA_RESET'), Config('API_2FA_ENABLE'), Config('API_2FA_DISABLE')])) {
                if ($this->security->isLoggedIn()) {
                    return true;
                } else {
                    $vote?->addReason(sprintf('Failed to perform 2FA action (%s). User is not logged in.', $actn));
                    return false;
                }
            }
        }

        // Default to true for other actions
        return true;
    }
}
