<?php

namespace PHPMaker2026\Reimbursement\Command;

use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Process\Process;

#[AsCommand(
    name: 'app:recipes',
    aliases: ['app:recipes:install', 'app:recipes:update'],
    description: 'Proxy for Composer Flex recipe commands (show, install, update) with Flex auto-toggle.'
)]
class RecipesCommand extends Command
{

    protected function configure(): void
    {
        // Allow arbitrary extra arguments like --force, -v, etc.
        $this->ignoreValidationErrors();
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $invoked = $_SERVER['argv'][1] ?? 'app:recipes';

        // Map Symfony command to Composer command
        $composerCmd = match (true) {
            str_contains($invoked, 'app:recipes:install') => 'recipes:install',
            str_contains($invoked, 'app:recipes:update')  => 'recipes:update',
            default                                       => 'recipes',
        };

        // Pass all arguments after our Symfony command
        $args = array_slice($_SERVER['argv'], 2);

        // Require a package name only for install/update
        if (in_array($composerCmd, ['recipes:install', 'recipes:update'], true) && empty($args)) {
            $output->writeln('<error>You must specify a package name, e.g. "symfony/asset-mapper".</error>');
            return Command::FAILURE;
        }
        $argsString = implode(' ', array_map(static function ($arg) {
            // Quote only when necessary
            return preg_match('/[\s"\']/', $arg) ? escapeshellarg($arg) : $arg;
        }, $args));

        // Build the actual Composer command
        $composer = trim(sprintf(
            'composer %s --ansi%s',
            $composerCmd,
            $argsString ? ' ' . $argsString : ''
        ));
        $output->writeln("<info>Running:</info> $composer\n");
        try {
            //------------------------------------------------------------------
            // Step 1: Enable Flex temporarily (silent)
            //------------------------------------------------------------------
            // $output->writeln('<comment>Temporarily enabling symfony/flex...</comment>');
            $enable = new Process(['composer', 'config', 'allow-plugins.symfony/flex', 'true']);
            $enable->disableOutput();
            $enable->setTimeout(10);
            $enable->run();
            if (!$enable->isSuccessful()) {
                $output->writeln('<error>Warning: Failed to enable symfony/flex.</error>');
            }

            //------------------------------------------------------------------
            // Step 2: Run main Composer command (visible, filtered)
            //------------------------------------------------------------------
            $process = Process::fromShellCommandline($composer);
            $process->setTimeout(null);
            $process->run(function ($type, $buffer) use ($output) {
                static $skipMissingBlock = false;
                foreach (explode("\n", $buffer) as $line) {
                    $trimmed = trim($line);
                    if ($trimmed === '') {
                        continue;
                    }

                    // Skip lines inside the """ ... """ missing-lines block
                    if ($skipMissingBlock) {
                        if (preg_match('/^"""/', $trimmed)) {
                            $skipMissingBlock = false;
                        }
                        continue;
                    }

                    // Detect start of "Could not add lines" for templates/base.html.twig only
                    if (preg_match('/^Could not add lines to file .*templates[\/\\\\]base\.html\.twig/i', $trimmed)) {
                        $skipMissingBlock = true;
                        continue;
                    }

                    // Filter out noisy Composer internals
                    if (preg_match(
                        '/^(Reading|Loading|Checked|Running|Executing|Writing|Generating|Installing|Skipped|Downloading|Fetching|\[\d{3}\])/i',
                        $trimmed
                    )) {
                        continue;
                    }
                    $output->writeln($line);
                }
            });
            if (!$process->isSuccessful()) {
                $output->writeln("\n<error>Command failed.</error>");
                return Command::FAILURE;
            }

            //------------------------------------------------------------------
            // Step 3: Fix twig_component.yaml namespace if exists
            //------------------------------------------------------------------
            $this->fixTwigComponentNamespace($output);
            return Command::SUCCESS;
        } catch (\Throwable $ex) {
            $output->writeln("<error>{$ex->getMessage()}</error>");
            return Command::FAILURE;
        } finally {
            //------------------------------------------------------------------
            // Step 4: Always restore Flex plugin state (silent)
            //------------------------------------------------------------------
            // $output->writeln('<comment>Restoring symfony/flex plugin configuration...</comment>');
            $disable = new Process(['composer', 'config', 'allow-plugins.symfony/flex', 'false']);
            $disable->disableOutput();
            $disable->setTimeout(10);
            $disable->run();
            if (!$disable->isSuccessful()) {
                $output->writeln('<error>Warning: Failed to restore symfony/flex state.</error>');
            }
        }
    }

    /**
     * Replace "App\" with current project namespace in twig_component.yaml.
     */
    protected function fixTwigComponentNamespace(OutputInterface $output): void
    {
        $file = dirname(__DIR__, 2) . '/config/packages/twig_component.yaml';
        if (!is_file($file)) {
            return;
        }
        $projectNamespace = preg_replace('/\\\\Command$/', '', __NAMESPACE__);
        $contents = file_get_contents($file);
        if (strpos($contents, 'App\\') === false) {
            return;
        }
        $updated = str_replace('App\\', $projectNamespace . '\\', $contents);
        if ($updated !== $contents) {
            file_put_contents($file, $updated);
            $output->writeln("<comment>Updated twig_component.yaml namespace to:</comment> $projectNamespace\\");
        }
    }
}
