<?php

namespace PHPMaker2026\Reimbursement;

use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Ldap\Exception\InvalidCredentialsException;
use Symfony\Component\Ldap\Exception\InvalidSearchCredentialsException;
use Symfony\Component\Ldap\LdapInterface;
use Symfony\Component\Ldap\Adapter\QueryInterface;
use Symfony\Component\Ldap\Adapter\EntryManagerInterface;

class CustomLdapWrapper implements LdapInterface
{

    public function __construct(
        private readonly LdapInterface $inner,
        private readonly RequestStack $requestStack,
        private readonly string $dnString,
        private readonly string $usernameParameter,
        private readonly string $passwordParameter,
        private readonly ?string $searchDn,
        #[\SensitiveParameter] private readonly ?string $searchPassword,
    ) {}

    public function bind(?string $dn = null, #[\SensitiveParameter] ?string $password = null): void
    {
        // Case 1: explicitly provided DN and password (can be empty strings)
        if ($dn && $password) {
            try {
                $this->inner->bind($dn, $password);
                return;
            } catch (InvalidCredentialsException) {
                // If using configured search credentials and failed, throw InvalidSearchCredentialsException
                if ($dn === $this->searchDn && $password === $this->searchPassword) {
                    throw new InvalidSearchCredentialsException();
                }
                // else fall through to anonymous bind
            }
        } else {
            // Case 2: try request-based bind
            $request = $this->requestStack->getCurrentRequest();
            if ($request !== null) {
                $username = $request->request->get($this->usernameParameter);
                $password = $request->request->get($this->passwordParameter);
                if ($username && $password) {
                    $resolvedDn = str_replace('{user_identifier}', $username, $this->dnString);
                    try {
                        $this->inner->bind($resolvedDn, $password);
                        return;
                    } catch (InvalidCredentialsException) {
                        // fall through to anonymous bind
                    }
                }
            }
        }

        // Final fallback: anonymous bind
        $this->inner->bind(null, null);
    }

    public function escape(string $subject, string $ignore = '', int $flags = 0): string
    {
        return $this->inner->escape($subject, $ignore, $flags);
    }

    public function query(string $dn, string $query, array $options = []): QueryInterface
    {
        return $this->inner->query($dn, $query, $options);
    }

    public function getEntryManager(): EntryManagerInterface
    {
        return $this->inner->getEntryManager();
    }
}
