<?php

namespace PHPMaker2026\Reimbursement\Db\Entity;

use DateTime;
use DateTimeImmutable;
use DateTimeInterface;
use DateInterval;
use Doctrine\ORM\Mapping\Column;
use Doctrine\ORM\Mapping\Entity;
use Doctrine\ORM\Mapping\GeneratedValue;
use Doctrine\ORM\Mapping\CustomIdGenerator;
use Doctrine\ORM\Mapping\Id;
use Doctrine\ORM\Mapping\Table;
use Doctrine\ORM\Mapping\SequenceGenerator;
use Doctrine\DBAL\Types\Types;
use Symfony\Component\Clock\DatePoint;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\EquatableInterface;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use PHPMaker2026\Reimbursement\AdvancedUserInterface;
use PHPMaker2026\Reimbursement\AdvancedSecurity;
use PHPMaker2026\Reimbursement\UserProfile;
use PHPMaker2026\Reimbursement\UserRepository;
use PHPMaker2026\Reimbursement\CustomEntityRepository;
use PHPMaker2026\Reimbursement\DefaultSequenceGenerator;
use PHPMaker2026\Reimbursement\UuidGenerator;
use PHPMaker2026\Reimbursement\Entity as BaseEntity;
use function PHPMaker2026\Reimbursement\Config;
use function PHPMaker2026\Reimbursement\EntityManager;
use function PHPMaker2026\Reimbursement\ConvertToBool;
use function PHPMaker2026\Reimbursement\ConvertToString;
use function PHPMaker2026\Reimbursement\SameDateTime;
use function PHPMaker2026\Reimbursement\RemoveXss;
use function PHPMaker2026\Reimbursement\HtmlDecode;
use function PHPMaker2026\Reimbursement\HashPassword;
use function PHPMaker2026\Reimbursement\PhpEncrypt;
use function PHPMaker2026\Reimbursement\PhpDecrypt;
use function PHPMaker2026\Reimbursement\Security;
use function PHPMaker2026\Reimbursement\IsEmpty;
use InvalidArgumentException;

/**
 * Entity class for 'mtuser' table
 */
#[Entity(repositoryClass: UserRepository::class)]
#[Table('mtuser')]
class Mtuser extends BaseEntity implements AdvancedUserInterface, EquatableInterface, PasswordAuthenticatedUserInterface
{
    #[Id]
    #[Column(name: 'ID', type: 'integer', unique: true, insertable: false, updatable: false)]
    #[GeneratedValue]
    private int $id;

    #[Column(name: 'Email', options: ['param' => '_Email'], type: 'string')]
    private string $email;

    #[Column(name: 'Password', options: ['password' => true], type: 'string', nullable: true)]
    private ?string $password;

    #[Column(name: 'FullName', type: 'string', nullable: true)]
    private ?string $fullName;

    #[Column(name: 'MTUserLevelID', type: 'integer')]
    private int $mtuserLevelId;

    public function getId(): int
    {
        return $this->id;
    }

    public function setId(int $value): static
    {
        $this->id = $value;
        return $this;
    }

    public function getEmail(): string
    {
        return $this->email;
    }

    public function setEmail(string $value): static
    {
        $this->email = $value;
        return $this;
    }

    public function getPassword(): ?string
    {
        return $this->password;
    }

    public function setPassword(?string $value): static
    {
        $this->password = $value;
        return $this;
    }

    public function getFullName(): ?string
    {
        return HtmlDecode($this->fullName);
    }

    public function setFullName(?string $value): static
    {
        $this->fullName = RemoveXss($value);
        return $this;
    }

    public function getMtuserLevelId(): int
    {
        return $this->mtuserLevelId;
    }

    public function setMtuserLevelId(int $value): static
    {
        $this->mtuserLevelId = $value;
        return $this;
    }

    /**
     * Get user ID
     *
     * @return mixed
     */
    public function userId(): mixed
    {
        return $this->get('ID');
    }

    /**
     * Get user name
     *
     * @return string
     */
    public function userName(): string
    {
        return $this->get('Email');
    }

    /**
     * Get parent user ID
     *
     * @return mixed
     */
    public function parentUserId(): mixed
    {
        return null;
    }

    /**
     * Get user level
     *
     * @return int|string
     */
    public function userLevel(): int|string
    {
        return $this->get('MTUserLevelID') ?? AdvancedSecurity::ANONYMOUS_USER_LEVEL_ID;
        ;
    }

    /**
     * Roles
     */
    protected array $roles = ['ROLE_USER'];

    /**
     * Get the roles granted to the user, e.g. ['ROLE_USER']
     *
     * @return string[]
     */
    public function getRoles(): array
    {
        $userLevelId = $this->get('MTUserLevelID');
        $roles = Security()->getAllRoles($userLevelId);
        return array_unique([...$this->roles, ...$roles]);
    }

    /**
     * Add a role
     *
     * @param string $role Role
     * @return void
     */
    public function addRole(string $role): void
    {
        if (!in_array($role, $this->roles)) {
            $this->roles[] = $role;
        }
    }

    /**
     * Remove a role
     *
     * @param string $role Role
     * @return void
     */
    public function removeRole(string $role): void
    {
        if (in_array($role, $this->roles)) {
            unset($this->roles[$role]);
        }
    }

    /**
     * Remove sensitive data
     */
    #[\Deprecated(since: 'symfony/security-core 7.3')]
    public function eraseCredentials(): void
    {
    }

    /**
     * Get user identifier (e.g. username or email address)
     */
    public function getUserIdentifier(): string
    {
        return $this->email;
    }

    /**
     * Compare users by attributes that are relevant for assessing whether re-authentication is required
     * See https://symfony.com/doc/current/security.html#understanding-how-users-are-refreshed-from-the-session
     */
    public function isEqualTo(UserInterface $user): bool
    {
        if (!$user instanceof self) {
            return false;
        }

        // if ($this->getPassword() !== $user->getPassword()) {
        //     return false;
        // }
        $currentRoles = array_map('strval', (array) $this->getRoles());
        $newRoles = array_map('strval', (array) $user->getRoles());
        $rolesChanged = count($currentRoles) !== count($newRoles) || count($currentRoles) !== count(array_intersect($currentRoles, $newRoles));
        if ($rolesChanged) {
            return false;
        }
        if ($this->getUserIdentifier() !== $user->getUserIdentifier()) {
            return false;
        }
        return true;
    }
}
