<?php

namespace PHPMaker2026\Reimbursement;

use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Session\Flash\FlashBagInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Security\Http\Event\LogoutEvent;
use Exception;

class LogoutListener implements EventSubscriberInterface
{
    protected FlashBagInterface $flash;
    protected LogoutEvent $event;

    public function __construct(
        protected UserProfile $profile,
        protected Language $language,
        protected AdvancedSecurity $security,
    ) {
    }

    public function getEvent(): LogoutEvent
    {
        return $this->event;
    }

    /**
     * Priorities of listeners of LogoutEvent:
     * DefaultLogoutListener (64)
     * THIS LISTENER (10) (BEFORE session being invalidated by SessionLogoutListener)
     * RememberMeListener
     * ClearSiteDataLogoutListener
     * CsrfTokenClearingLogoutListener
     * SessionLogoutListener (if 'logout.invalidate_session' enabled)
     * CookieClearingLogoutListener (-255)
     *
     * @return array
     */
    public static function getSubscribedEvents(): array
    {
        return [
            LogoutEvent::class => ['onLogout', 10]
        ];
    }

    protected function setNotice(string $type, mixed $message)
    {
        if (!$this->flash->has('heading')) {
            $this->flash->add('heading', $this->language->phrase('Notice'));
        }
        if (!$this->flash->has($type)) {
            $this->flash->add($type, $message);
        }
    }

    public function onLogout(LogoutEvent $event): void
    {
        $this->event = $event;
        $request = $event->getRequest();
        $session = $request->getSession();
        $sessionId = $session->getId();
        $user = $event->getToken()?->getUser();
        $username = $user?->getUserIdentifier() ?? "";

        // Call User LoggingOut event
        $this->userLoggingOut($username);
        $params = $request->query->all();
        $this->flash = $session->getFlashBag();

        // Remove last URL
        $this->security->removeLastUrl();

        // Password changed (after expired password)
        if (Config('USE_TWO_FACTOR_AUTHENTICATION') && $session->get(SESSION_STATUS) == 'passwordchanged') {
            $request->attributes->set('_success_message', 'LoginAfterPasswordChanged');
        }

        // Call User LoggedOut event
        $this->userLoggedOut($username);

        // Clean upload temp folder
        CleanUploadTempPaths($sessionId);

        // Clear user profile
        $this->security->logout();
    }

    // User Logging Out event
    public function userLoggingOut(string $userName): void
    {
        //Log("User Logging Out");
    }

    // User Logged Out event
    public function userLoggedOut(string $userName): void
    {
        //Log("User Logged Out");
    }
}
