<?php

namespace PHPMaker2026\Reimbursement;

use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authorization\Voter\Vote;
use Symfony\Component\Security\Core\Authorization\Voter\Voter as BaseVoter;

/**
 * Permission Voter
 */
class PermissionVoter extends BaseVoter
{

    public function __construct(
        protected RequestStack $requestStack,
        protected AdvancedSecurity $security,
        protected UserProfile $profile,
    ) {
    }

    protected function supports(string $attribute, mixed $subject): bool
    {
        $request = $this->requestStack->getCurrentRequest();
        return !str_starts_with($request->getPathInfo(), '/api/');
    }

    protected function voteOnAttribute(string $attribute, mixed $subject, TokenInterface $token, ?Vote $vote = null): bool
    {
        $request = $this->requestStack->getCurrentRequest();
        $routeName = $request->attributes->get('_route', '');

        // Get page action and table
        [$pageAction, $table, $name] = explode('.', $routeName) + [null, null, null]; // Make sure at least three elements

        // Security
        $security = $this->security;

        // Login
        if (IsLoggingIn2FA()) {
            if ($user = CurrentUser()) {
                $this->profile->setUserName($user->getUserIdentifier());
                $security->loginUser($this->profile); // Login user with user name only (no permissions yet)
            } else { // Token was deauthenticated after trying to refresh it
                $request->getSession()->remove(SESSION_STATUS);
                $vote?->addReason('Token was deauthenticated after trying to refresh it');
                return false;
            }
        } elseif (
            !$security->isLoggedIn()
            && !IsPasswordReset()
            && !IsPasswordExpired()
        ) {
            $security->login();
        }

        // Load user level
        $security->initialize();

        // Check permission
        if ($table != '') { // Table level
            $security->loadTablePermissions($table);
            if (
                $pageAction == Config('VIEW_ACTION') && !$security->canView()
                || in_array($pageAction, [Config('EDIT_ACTION'), Config('UPDATE_ACTION')]) && !$security->canEdit()
                || $pageAction == Config('ADD_ACTION') && !$security->canAdd()
                || $pageAction == Config('DELETE_ACTION') && !$security->canDelete()
                || in_array($pageAction, [Config('SEARCH_ACTION'), Config('QUERY_ACTION')]) && !$security->canSearch()
            ) {
                if ($security->canList()) { // Back to list
                    $pageAction = Config('LIST_ACTION');
                    $routeUrl = Container($table)->getListUrl();
                    $request->attributes->set('redirect', $pageAction . '.' . $table);
                }
                $vote?->addReason('User does not have "' . $pageAction . '" permission.');
                return false;
            } elseif (
                $pageAction == Config('LIST_ACTION') && !$security->canList() // List Permission
                || in_array($pageAction, [
                    Config('CUSTOM_REPORT_ACTION'),
                    Config('SUMMARY_REPORT_ACTION'),
                    Config('CROSSTAB_REPORT_ACTION'),
                    Config('DASHBOARD_REPORT_ACTION'),
                    Config('CALENDAR_REPORT_ACTION')
                ]) && !$security->canList()
            ) { // No permission
                $vote?->addReason('User does not have "List" permission.');
                return false;
            }
        } else { // Others
            if ($pageAction == 'changepassword') { // Change password
                if (!IsPasswordReset() && !IsPasswordExpired()) {
                    if (!$security->isLoggedIn() || $security->isSysAdmin()) {
                        $vote?->addReason('User is not logged in or is super admin.');
                        return false;
                    }
                }
            } elseif ($pageAction == 'personaldata') { // Personal data
                if (!$security->isLoggedIn() || $security->isSysAdmin()) {
                    $vote?->addReason('User is not logged in or is super admin.');
                    return false;
                }
            } elseif ($pageAction == 'userpriv') { // User priv
                $table = 'mtuserlevel';
                $pageAction = Config('LIST_ACTION');
                $routeUrl = Container($table)->getListUrl();
                $security->loadTablePermissions($table);
                if (!$security->isLoggedIn() || !$security->canGrant()) {
                    $request->attributes->set('redirect', $pageAction . '.' . $table);
                    $vote?->addReason('User is not logged in or does not have "Grant" permission.');
                    return false;
                }
            }
        }
        return true;
    }
}
