<?php

namespace PHPMaker2026\Reimbursement;

use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Session\Flash\FlashBagInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Security\Http\Event\LogoutEvent;
use Exception;

class PostLogoutListener implements EventSubscriberInterface
{
    protected FlashBagInterface $flash;

    public function __construct(
        protected Language $language,
    ) {
    }

    /**
     * Priorities of listeners of LogoutEvent:
     * DefaultLogoutListener (64)
     * RememberMeListener
     * ClearSiteDataLogoutListener
     * CsrfTokenClearingLogoutListener
     * SessionLogoutListener (if 'logout.invalidate_session' enabled)
     * THIS LISTENER (-100) (AFTER session being invalidated by SessionLogoutListener)
     * CookieClearingLogoutListener (-255)
     *
     * @return array
     */
    public static function getSubscribedEvents(): array
    {
        return [
            LogoutEvent::class => ['onLogout', -100]
        ];
    }

    protected function setNotice(string $type, mixed $message)
    {
        if (!$this->flash->has('heading')) {
            $this->flash->add('heading', $this->language->phrase('Notice'));
        }
        if (!$this->flash->has($type)) {
            $this->flash->add($type, $message);
        }
    }

    public function onLogout(LogoutEvent $event): void
    {
        $request = $event->getRequest();
        $session = $request->getSession();
        $this->flash = $session->getFlashBag();

        // Handle flash messages for success, failure, warning, and info
        foreach (['success', 'danger', 'warning', 'info'] as $type) {
            $attr = "_{$type}_message";
            if ($request->attributes->has($attr)) {
                $this->setNotice($type, $this->language->phrase($request->attributes->get($attr)));
            }
        }

        // If session expired, show expired message
        if ($request->attributes->has('expired')) {
            $this->setNotice('danger', $this->language->phrase('SessionExpired'));
        }

        // Go to login page
        $url = UrlFor('login');
        $this->pageRedirecting($url);
        if (IsJsonResponse() || IsModal()) {
            $event->setResponse(new JsonResponse(['url' => $url]));
            return;
        }
        $event->setResponse(new RedirectResponse($url));
    }

    // Page Redirecting event
    public function pageRedirecting(?string &$url): void
    {
        // Example:
        //$url = "your URL";
    }
}
