<?php

namespace Lexik\Bundle\JWTAuthenticationBundle\Tests\DependencyInjection;

use Lexik\Bundle\JWTAuthenticationBundle\DependencyInjection\LexikJWTAuthenticationExtension;
use Lexik\Bundle\JWTAuthenticationBundle\Encoder\LcobucciJWTEncoder;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\Compiler\ResolveChildDefinitionsPass;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Tests the bundle extension and the configuration of services.
 *
 * @author Robin Chalas <robin.chalas@gmail.com>
 */
class LexikJWTAuthenticationExtensionTest extends TestCase
{
    public function testEncoderConfiguration()
    {
        $container = $this->getContainer(['secret_key' => 'private.pem', 'public_key' => 'public.pem', 'additional_public_keys' => ['public1.pem', 'public2.pem'], 'pass_phrase' => 'test']);
        $encoderDef = $container->findDefinition('lexik_jwt_authentication.encoder');
        $this->assertSame(LcobucciJWTEncoder::class, $encoderDef->getClass());
        $this->assertEquals(new Reference('lexik_jwt_authentication.jws_provider.lcobucci'), $encoderDef->getArgument(0));
        $this->assertEquals(
            [
                new Reference('lexik_jwt_authentication.key_loader.raw'),
                '%lexik_jwt_authentication.encoder.signature_algorithm%',
                '%lexik_jwt_authentication.token_ttl%',
                '%lexik_jwt_authentication.clock_skew%',
                '%lexik_jwt_authentication.allow_no_expiration%',
            ],
            $container->getDefinition('lexik_jwt_authentication.jws_provider.lcobucci')->getArguments()
        );
        $this->assertSame(
            ['private.pem', 'public.pem', '%lexik_jwt_authentication.pass_phrase%', ['public1.pem', 'public2.pem']],
            $container->getDefinition('lexik_jwt_authentication.key_loader.raw')->getArguments()
        );
    }

    public function testTokenExtractorsConfiguration()
    {
        // Default configuration
        $chainTokenExtractor = $this->getContainer(['secret_key' => 'private.pem', 'public_key' => 'public.pem'])->getDefinition('lexik_jwt_authentication.extractor.chain_extractor');

        $extractorIds = array_map('strval', $chainTokenExtractor->getArgument(0));

        $this->assertContains('lexik_jwt_authentication.extractor.authorization_header_extractor', $extractorIds);
        $this->assertNotContains('lexik_jwt_authentication.extractor.cookie_extractor', $extractorIds);
        $this->assertNotContains('lexik_jwt_authentication.extractor.query_parameter_extractor', $extractorIds);

        // Custom configuration
        $chainTokenExtractor = $this->getContainer(['secret_key' => 'private.pem', 'public_key' => 'public.pem', 'token_extractors' => ['authorization_header' => true, 'cookie' => true]])
            ->getDefinition('lexik_jwt_authentication.extractor.chain_extractor');

        $extractorIds = array_map('strval', $chainTokenExtractor->getArgument(0));

        $this->assertContains('lexik_jwt_authentication.extractor.authorization_header_extractor', $extractorIds);
        $this->assertContains('lexik_jwt_authentication.extractor.cookie_extractor', $extractorIds);
        $this->assertNotContains('lexik_jwt_authentication.extractor.query_parameter_extractor', $extractorIds);
    }

    private function getContainer($config = [])
    {
        $container = new ContainerBuilder();
        $container->registerExtension(new LexikJWTAuthenticationExtension());
        $container->loadFromExtension('lexik_jwt_authentication', $config);

        $container->getCompilerPassConfig()->setOptimizationPasses([new ResolveChildDefinitionsPass()]);
        $container->getCompilerPassConfig()->setRemovingPasses([]);
        $container->getCompilerPassConfig()->setAfterRemovingPasses([]);
        $container->compile();

        return $container;
    }
}
